--[[
    VehicleConfigurationItemColorExtension

    Extension to VehicleConfigurationItemColor to add support for agriBumperColor configuration and prevent unwanted warnings.

	@author: 		BayernGamers
	@date: 			26.10.2025
	@version:		1.0

	History:		v1.0 @26.10.2025 - initial implementation in FS 25
                    ------------------------------------------------------------------------------------------------------
	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "VehicleConfigurationItemColorExtension.lua")

VehicleConfigurationItemColorExtension = {}

function VehicleConfigurationItemColorExtension:onPostLoad(superFunc, object, configId)
    VehicleConfigurationItemColor:superClass().onPostLoad(self, object, configId)
    
    local configurationDesc = g_vehicleConfigurationManager:getConfigurationDescByName(self.configName)

    if configurationDesc.configurationsKey:find("agriBumperColor") == nil then
        return superFunc(self, object, configId)
    end

    object.xmlFile:iterate(configurationDesc.configurationsKey.. ".material", function(index, materialKey)
        local material = VehicleMaterial.new(object.baseDirectory)

        local color, materialTemplateName = self:getColorAndMaterialFromVehicle(object)

        if object.xmlFile:getValue(materialKey .. "#useContrastColor", false) then
            local threshold = object.xmlFile:getValue(materialKey .. "#contrastThreshold", 0.5)
            local brightness = MathUtil.getBrightnessFromColor(color[1], color[2], color[3])
            if brightness > threshold then
                color = object.xmlFile:getValue(materialKey .. "#contrastColorDark", "0 0 0", true)
            else
                color = object.xmlFile:getValue(materialKey .. "#contrastColorBright", "0.9 0.9 0.9", true)
            end
        end

        if object.xmlFile:getValue(materialKey .. "#materialTemplateName", nil) == nil then
            if not object.xmlFile:getValue(materialKey .. "#materialTemplateUseColorOnly", false) then
                material:setTemplateName(materialTemplateName, nil, object.customEnvironment)
            end

            material:setColor(color)
        end

        material:loadFromXML(object.xmlFile, materialKey, object.customEnvironment)
        if material.targetMaterialSlotName ~= nil then
            if not material:applyToVehicle(object) then
                if materialKey:find("agriBumperColor") == nil then
                    Logging.xmlWarning(object.xmlFile, "Failed to find material by material slot name '%s' in '%s'", material.targetMaterialSlotName, materialKey)
                else
                    log:printDevXMLWarning(object.xmlFile, string.format("Failed to find material by material slot name '%s' in '%s'", material.targetMaterialSlotName, materialKey), LoggingUtil.DEBUG_LEVELS.HIGH)
                end
            end
        else
            Logging.xmlWarning(object.xmlFile, "Missing material slot name in '%s'", materialKey)
        end
    end)
end

VehicleConfigurationItemColor.onPostLoad = Utils.overwrittenFunction(VehicleConfigurationItemColor.onPostLoad, VehicleConfigurationItemColorExtension.onPostLoad)