--[[
    ExtendSpecializations

    Extends the specializations of vehicle types based on the modDesc.xml configuration.

	@author: 		BayernGamers
	@date: 			04.09.2025
	@version:		1.1

	History:		v1.0 @12.11.2024 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------
                    v1.1 @04.09.2025 - adjustments in order to promote extensions to parents as well

	
	License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "ExtendSpecializations.lua")

ExtendSpecializations = {}
ExtendSpecializations.debugLevelHigh = 2
ExtendSpecializations.debugLevel = 1
ExtendSpecializations.MOD_DIRECTORY = g_currentModDirectory
ExtendSpecializations.MOD_NAME = g_currentModName
ExtendSpecializations.XML_KEY = "modDesc.extendSpecializations.vehicleType"
ExtendSpecializations.modDescPath = ExtendSpecializations.MOD_DIRECTORY .. "modDesc.xml"
ExtendSpecializations.specializations = {}

local modDesc = XMLFile.load("ModFile", ExtendSpecializations.modDescPath)

local function extendSpecialization(vehicleTypeName, vehicleType, key)
    if vehicleType.specializations ~= nil then
        modDesc:iterate(key .. ".specialization", function(_, specKey)
            local specName = modDesc:getString(specKey .. "#name")

            if g_specializationManager.specializations[specName] ~= nil and vehicleType.specializationsByName[specName] == nil then
                log:printDevInfo("Extending vehicleType '" .. vehicleTypeName .. "' with specialization '" .. specName .. "'", ExtendSpecializations.debugLevelHigh)
                g_vehicleTypeManager:addSpecialization(vehicleTypeName, specName)
            elseif vehicleType.specializationsByName[specName] ~= nil then
                log:printDevInfo("VehicleType '" .. vehicleTypeName .. "' already has specialization '" .. specName .. "'", ExtendSpecializations.debugLevelHigh)
            else
                log:printError("Unable to find specialization '" .. specName .. "'")
            end
        end)
    else
        log:printDevWarning("VehicleType '" .. vehicleTypeName .. "' already has specialization '" .. specName .. "'", ExtendSpecializations.debugLevelHigh)
    end
end

modDesc:iterate(ExtendSpecializations.XML_KEY, function (_, key)
    local typeName = modDesc:getString(key .. "#name")
    log:printDevInfo("Extending Specializations for vehicleType '" .. typeName .. "'", ExtendSpecializations.debugLevelHigh)

    for vehicleTypeName, vehicleType in pairs(g_vehicleTypeManager.types) do
        if vehicleType ~= nil and vehicleTypeName == typeName then
            extendSpecialization(vehicleTypeName, vehicleType, key)
        end

        local function isChildOfType(vType, parentName)
            local current = vType.parent
            while current ~= nil do
                if current.name == parentName then
                    return true
                end
                current = current.parent
            end
            return false
        end

        if vehicleType ~= nil and vehicleTypeName ~= typeName and isChildOfType(vehicleType, typeName) then
            log:printDevInfo("VehicleType '" .. vehicleTypeName .. "' is a child of '" .. typeName .. "'. Extending as well.", ExtendSpecializations.debugLevelHigh)
            extendSpecialization(vehicleTypeName, vehicleType, key)
        end
    end
end)

modDesc:delete()