--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 05.10.2019
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Free for use in mods (FS25 Only) - no permission needed.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy

Frei verwendbar (Nur LS25) - keine erlaubnis nötig
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
]]

PressureWasherVehicle = {}

PressureWasherVehicle.MOD_NAME = g_currentModName
PressureWasherVehicle.BASE_DIRECTORY = g_currentModDirectory
PressureWasherVehicle.SPEC_NAME = string.format("spec_%s.pressureWasherVehicle", PressureWasherVehicle.MOD_NAME)

function PressureWasherVehicle.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(HighPressureWasher, specializations) then
        Logging.error("Vehicle specialization '%s' is not required when using 'PressureWasherVehicle' specialization.")

        return false
    end

    return SpecializationUtil.hasSpecialization(HandToolHolders, specializations)
end

function PressureWasherVehicle.initSpecialization()
    if g_vehicleConfigurationManager:getConfigurationDescByName("pressureWasherConsumer") == nil then
        g_vehicleConfigurationManager:addConfigurationType("pressureWasherConsumer", g_i18n:getText("configuration_consumer"), "pressureWasher", VehicleConfigurationItem)
    end

    local schema = Vehicle.xmlSchema

    schema:setXMLSpecializationType("PressureWasherVehicle")

    schema:register(XMLValueType.INT, "vehicle.pressureWasher.pump#maxPressure", "Maximum pump pressure (PSI)", 4000)
    -- schema:register(XMLValueType.BOOL, "vehicle.pressureWasher.pump#canAdjustPressure", "Allows supported lance to adjust pressure", false)

    local consumerConfigurationKey = "vehicle.pressureWasher.pressureWasherConsumerConfigurations.pressureWasherConsumerConfiguration(?)"

    schema:register(XMLValueType.FLOAT, consumerConfigurationKey .. ".consumer(?)#fillUnitIndex", "Consumer fill unit index", 1)
    schema:register(XMLValueType.STRING, consumerConfigurationKey .. ".consumer(?)#fillType", "Fill type name")
    schema:register(XMLValueType.FLOAT, consumerConfigurationKey .. ".consumer(?)#usage", "Usage in l/h", 1)
    schema:register(XMLValueType.FLOAT, consumerConfigurationKey .. ".consumer(?)#washMultiplierIncrease", "Extra value applied when fill type is available. Excludes 'Water' & 'Diesel'", 0.01)

    ObjectChangeUtil.registerObjectChangeXMLPaths(schema, consumerConfigurationKey)

    -- schema:register(XMLValueType.NODE_INDEX, "vehicle.pressureWasher.hoseConnection#node", "Connection point to calculate the hose length used and dynamic connection", "0>")

    AnimationManager.registerAnimationNodesXMLPaths(schema, "vehicle.pressureWasher.animationNodes")
    EffectManager.registerEffectXMLPaths(schema, "vehicle.pressureWasher.effects")

    SoundManager.registerSampleXMLPaths(schema, "vehicle.pressureWasher.sounds", "start(?)")
    SoundManager.registerSampleXMLPaths(schema, "vehicle.pressureWasher.sounds", "stop(?)")
    SoundManager.registerSampleXMLPaths(schema, "vehicle.pressureWasher.sounds", "work(?)")

    schema:register(XMLValueType.STRING, "vehicle.pressureWasher.turnedAnimation#name", "Turned animation name (Animation played while activating and deactivating)")
    schema:register(XMLValueType.FLOAT, "vehicle.pressureWasher.turnedAnimation#turnOnSpeedScale", "Turn on speed scale", 1)
    schema:register(XMLValueType.FLOAT, "vehicle.pressureWasher.turnedAnimation#turnOffSpeedScale", "Turn off speed scale", "Inversed turnOnSpeedScale")

    schema:register(XMLValueType.FLOAT, "vehicle.pressureWasher.startDuration", "Duration pump motor takes to start", "Start sample time")
    schema:register(XMLValueType.FLOAT, "vehicle.pressureWasher.dashboards#delayedStartDuration", "Allows dashboard groups to have a delayed start time", 0)

    ObjectChangeUtil.registerObjectChangeXMLPaths(schema, "vehicle.pressureWasher.holderObjectChanges")
    ObjectChangeUtil.registerObjectChangeXMLPaths(schema, "vehicle.pressureWasher.objectChanges")

    Dashboard.registerDashboardXMLPaths(schema, "vehicle.pressureWasher.dashboards", {"time", "operatingTime", "waterPressure", "waterFlowRate"})

    schema:register(XMLValueType.BOOL, Dashboard.GROUP_XML_KEY .. "#isPressureWasherRunning", "Is running")
    schema:register(XMLValueType.BOOL, Dashboard.GROUP_XML_KEY .. "#isPressureWasherStarting", "Is starting")
    schema:register(XMLValueType.BOOL, Dashboard.GROUP_XML_KEY .. "#isPressureWasherDelayedStart", "Is delayed start")

    schema:setXMLSpecializationType()

    -- local schemaSavegame = Vehicle.xmlSchemaSavegame
    -- schemaSavegame:register(XMLValueType.FLOAT, string.format("vehicles.vehicle(?).%s.pressureWasherVehicle#pumpWorkMultiplier", PressureWasherVehicle.MOD_NAME), "Pump work multiplier", 1)
end

function PressureWasherVehicle.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", PressureWasherVehicle)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterDashboardValueTypes", PressureWasherVehicle)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", PressureWasherVehicle)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", PressureWasherVehicle)
    SpecializationUtil.registerEventListener(vehicleType, "onHandToolStoredInHolder", PressureWasherVehicle)
    SpecializationUtil.registerEventListener(vehicleType, "onHandToolTakenFromHolder", PressureWasherVehicle)
    SpecializationUtil.registerEventListener(vehicleType, "onFillUnitFillLevelChanged", PressureWasherVehicle) -- Remove if there is no consumers TO_DO
end

function PressureWasherVehicle.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "loadDashboardGroupFromXML", PressureWasherVehicle.loadDashboardGroupFromXML)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getIsDashboardGroupActive", PressureWasherVehicle.getIsDashboardGroupActive)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getIsOperating", PressureWasherVehicle.getIsOperating)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "showInfo", PressureWasherVehicle.showInfo)
end

function PressureWasherVehicle.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "setPressureWasherActive", PressureWasherVehicle.setPressureWasherActive)
    SpecializationUtil.registerFunction(vehicleType, "setPressureWasherIsWashing", PressureWasherVehicle.setPressureWasherIsWashing)
    SpecializationUtil.registerFunction(vehicleType, "updatePressureWasherConsumers", PressureWasherVehicle.updatePressureWasherConsumers)
    SpecializationUtil.registerFunction(vehicleType, "getCanPressureWasherRun", PressureWasherVehicle.getCanPressureWasherRun)
    SpecializationUtil.registerFunction(vehicleType, "getPressureWasherActive", PressureWasherVehicle.getPressureWasherActive)
    SpecializationUtil.registerFunction(vehicleType, "getFormattedPressureWasherOperatingTime", PressureWasherVehicle.getFormattedPressureWasherOperatingTime)
    SpecializationUtil.registerFunction(vehicleType, "getPressureWasherEngineLoad", PressureWasherVehicle.getPressureWasherEngineLoad)
    SpecializationUtil.registerFunction(vehicleType, "getPressureWasherWorkMultiplier", PressureWasherVehicle.getPressureWasherWorkMultiplier)
end

function PressureWasherVehicle:onLoad(savegame)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec == nil then
        Logging.error("[%s] Specialization with name 'pressureWasherVehicle' was not found in modDesc!", PressureWasherVehicle.MOD_NAME)
    end

    local xmlFile = self.xmlFile

    spec.isActive = false
    spec.isWashing = false

    spec.engineLoad = 0
    spec.startTime = 0
    spec.delayedStartTime = 0

    spec.pump = {
        pressure = 0,
        flowRate = 0,
        maxFlowRate = 0,
        workMultiplier = 1,
        maxPressure = xmlFile:getValue("vehicle.pressureWasher.pump#maxPressure", 4000)
        -- canAdjustPressure = xmlFile:getValue("vehicle.pressureWasher.pump#canAdjustPressure", false)
    }

    spec.consumers = {}
    spec.consumersByFillTypeIndex = {}

    local firstFillTheToolWarning = g_i18n:getText("info_firstFillTheTool")
    local consumerConfigurationId = Utils.getNoNil(self.configurations["pressureWasherConsumer"], 1)
    local consumerConfigurationKey = string.format("vehicle.pressureWasher.pressureWasherConsumerConfigurations.pressureWasherConsumerConfiguration(%d)", consumerConfigurationId - 1)

    ObjectChangeUtil.updateObjectChanges(xmlFile, "vehicle.pressureWasher.pressureWasherConsumerConfigurations.pressureWasherConsumerConfiguration", consumerConfigurationId , self.components, self)

    -- Consumers are optional. Diesel and Water if used will be required for operation.
    -- All other fillTypes are treated as additives and will speed up wash times using 'washMultiplierIncrease' only if fill level > 0

    xmlFile:iterate(consumerConfigurationKey .. ".consumer", function (_, consumerKey)
        local fillUnitIndex = xmlFile:getValue(consumerKey .. "#fillUnitIndex", 1)
        local fillUnit = self:getFillUnitByIndex(fillUnitIndex)

        if fillUnit ~= nil and fillUnit.capacity > 0 then
            local fillTypeName = xmlFile:getValue(consumerKey .. "#fillType", "consumer"):upper()
            local fillType = g_fillTypeManager:getFillTypeByName(fillTypeName)

            if fillType ~= nil then
                local fillTypeIndex = fillType.index

                if spec.consumersByFillTypeIndex[fillTypeIndex] == nil then
                    local consumer = {
                        fillLevel = fillUnit.fillLevel,
                        title = fillType.title,
                        fillTypeIndex = fillTypeIndex,
                        fillUnitIndex = fillUnitIndex,
                        washMultiplierIncrease = 0,
                        isRequired = true
                    }

                    consumer.emptyWarning = string.format("%s ( %s )", firstFillTheToolWarning, consumer.title)
                    consumer.usagePerMS = math.max(xmlFile:getValue(consumerKey .. "#usage", 1000), 1) / (60 * 60 * 1000)

                    if fillTypeName == "WATER" then
                        spec.pump.maxFlowRate = consumer.usagePerMS * (60 * 1000)
                    elseif fillTypeName ~= "DIESEL" then
                        consumer.washMultiplierIncrease = math.max(xmlFile:getValue(consumerKey .. "#washMultiplierIncrease", 0.1), 0.01)
                        consumer.isRequired = false
                    end

                    table.insert(spec.consumers, consumer)

                    spec.consumersByFillTypeIndex[consumer.fillTypeIndex] = consumer
                else
                    Logging.xmlWarning(xmlFile, "Consumer with fillType '%s' already in use!", fillTypeName)
                end
            else
                Logging.xmlWarning(xmlFile, "Unknown fillType '%s' for consumer '%s'", fillTypeName, consumerKey)
            end
        else
            Logging.xmlError(xmlFile, "FillUnit '%d' not defined or capacity is 0 at '%s'!", fillUnitIndex, consumerKey)
        end
    end)

    spec.numberOfConsumer = #spec.consumers
    -- spec.hoseConnectionNode = xmlFile:getValue("vehicle.pressureWasher.hoseConnection#node", "0>", self.components, self.i3dMappings)

    if self.isClient then
        spec.animationNodes = g_animationManager:loadAnimations(self.xmlFile, "vehicle.pressureWasher.animationNodes", self.components, self, self.i3dMappings)
        spec.effects = g_effectManager:loadEffect(self.xmlFile, "vehicle.pressureWasher.effects", self.components, self, self.i3dMappings)

        spec.samples = {
            start = g_soundManager:loadSampleFromXML(xmlFile, "vehicle.pressureWasher.sounds", "start", self.baseDirectory, self.components, 1, AudioGroup.VEHICLE, self.i3dMappings, self),
            stop = g_soundManager:loadSampleFromXML(xmlFile, "vehicle.pressureWasher.sounds", "stop", self.baseDirectory, self.components, 1, AudioGroup.VEHICLE, self.i3dMappings, self),
            work = g_soundManager:loadSamplesFromXML(xmlFile, "vehicle.pressureWasher.sounds", "work", self.baseDirectory, self.components, 0, AudioGroup.VEHICLE, self.i3dMappings, self)
        }

        if SpecializationUtil.hasSpecialization(AnimatedVehicle, self.specializations) then
            local turnOnAnimationName = xmlFile:getValue("vehicle.pressureWasher.turnedAnimation#name")

            if turnOnAnimationName ~= nil then
                spec.turnOnAnimation = {
                    name = turnOnAnimationName,
                    turnOnSpeedScale = xmlFile:getValue("vehicle.pressureWasher.turnedAnimation#turnOnSpeedScale", 1)
                }

                spec.turnOnAnimation.turnOffSpeedScale = xmlFile:getValue("vehicle.pressureWasher.turnedAnimation#turnOffSpeedScale", -spec.turnOnAnimation.turnOnSpeedScale)
            end
        end

        spec.holderObjectChanges = {}
        ObjectChangeUtil.loadObjectChangeFromXML(self.xmlFile, "vehicle.pressureWasher.holderObjectChanges", spec.holderObjectChanges, self.components, self)

        spec.objectChanges = {}
        ObjectChangeUtil.loadObjectChangeFromXML(self.xmlFile, "vehicle.pressureWasher.objectChanges", spec.objectChanges, self.components, self)
    end

    spec.startDuration = xmlFile:getValue("vehicle.pressureWasher.startDuration", spec.samples.start ~= nil and spec.samples.start.duration or 0)
    spec.delayedStartDuration = xmlFile:getValue("vehicle.pressureWasher.dashboards#delayedStartDuration", 0) * 1000
end

function PressureWasherVehicle:onRegisterDashboardValueTypes()
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec ~= nil then
        local time = DashboardValueType.new("pressureWasher", "time")
        time:setValue(g_currentMission.environment, "getEnvironmentTime")
        self:registerDashboardValueType(time)

        local operatingTime = DashboardValueType.new("pressureWasher", "operatingTime")
        operatingTime:setValue(self, self.getFormattedPressureWasherOperatingTime)
        self:registerDashboardValueType(operatingTime)

        local waterPressure = DashboardValueType.new("pressureWasher", "waterPressure")
        waterPressure:setRange(0, spec.pump.maxPressure)
        waterPressure:setValue(self, function()
            return spec.pump.pressure
        end)
        waterPressure:setInterpolationSpeed(function()
            return spec.pump.maxPressure * 0.001
        end)
        self:registerDashboardValueType(waterPressure)

        if spec.numberOfConsumer > 0 then
            local waterPressure = DashboardValueType.new("pressureWasher", "waterFlowRate")
            waterPressure:setRange(0, spec.pump.maxFlowRate)
            waterPressure:setValue(self, function()
                return spec.pump.flowRate
            end)
            waterPressure:setInterpolationSpeed(function()
                return spec.pump.maxFlowRate * 0.001
            end)
            self:registerDashboardValueType(waterPressure)
        end
    end
end

function PressureWasherVehicle:onDelete()
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    self:setPressureWasherActive(false)

    if self.isClient then
        g_animationManager:deleteAnimations(spec.animationNodes)
        g_effectManager:deleteEffects(spec.effects)

        if spec.samples ~= nil then
            g_soundManager:deleteSample(spec.samples.start)
            g_soundManager:deleteSample(spec.samples.stop)
            g_soundManager:deleteSamples(spec.samples.work)
        end

        g_effectManager:deleteEffects(spec.effects)
    end
end

function PressureWasherVehicle:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec.handTool ~= nil and not spec.isActive then
        if self:getCanPressureWasherRun() then
            self:setPressureWasherActive(true)
        end

        self:raiseActive()
    end

    if spec.isActive then
        if spec.engineLoad < 1 and spec.startTime <= g_currentMission.time then
            spec.engineLoad = math.min(spec.engineLoad + 0.01, 1)
        end

        if spec.isWashing then
            spec.pump.flowRate = (spec.pump.maxFlowRate * spec.pump.workMultiplier) * spec.engineLoad
            spec.pump.pressure = (spec.pump.maxPressure * spec.pump.workMultiplier) * spec.engineLoad
        else
            spec.pump.flowRate = 0
            spec.pump.pressure = 0
        end

        if self.isClient then
            g_effectManager:setDensity(spec.effects, spec.engineLoad)
        end

        self:raiseActive()
    elseif spec.engineLoad > 0 then
        spec.engineLoad = math.max(spec.engineLoad - 0.02, 0)
        self:raiseActive()
    end
end

function PressureWasherVehicle:onHandToolTakenFromHolder(handTool)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if handTool ~= nil and handTool.setHighPressureWasher ~= nil then
        handTool:setHighPressureWasher(self)
    end

    spec.handTool = handTool

    if self:getCanPressureWasherRun() then
        self:setPressureWasherActive(true)
    else
        self:raiseActive()
    end

    if self.isClient then
        ObjectChangeUtil.setObjectChanges(spec.holderObjectChanges, true, self, nil, true)
    end
end

function PressureWasherVehicle:onHandToolStoredInHolder(handTool)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if handTool ~= nil and handTool.setHighPressureWasher ~= nil then
        handTool:setHighPressureWasher(nil)
    end

    spec.handTool = nil
    self:setPressureWasherActive(false)

    if self.isClient then
        ObjectChangeUtil.setObjectChanges(spec.holderObjectChanges, false, self, nil, true)
    end
end

function PressureWasherVehicle:onFillUnitFillLevelChanged(fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData, appliedDelta)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec ~= nil then
        local consumer = spec.consumersByFillTypeIndex[fillTypeIndex]

        if consumer ~= nil then
            consumer.fillLevel = self:getFillUnitFillLevel(fillUnitIndex) or 0

            if spec.isActive and fillTypeIndex == FillType.DIESEL and consumer.fillLevel <= 0 then
                self:setPressureWasherActive(false)
            end
        end
    end
end

function PressureWasherVehicle:updatePressureWasherConsumers(dt, isWashing, washMultiplier)
    washMultiplier = washMultiplier or 1

    if self.isServer then
        local spec = self[PressureWasherVehicle.SPEC_NAME]

        if spec ~= nil and spec.numberOfConsumer > 0 then
            local pumpWorkMultiplier = spec.pump.workMultiplier
            local farmId = self:getOwnerFarmId()

            for _, consumer in ipairs (spec.consumers) do
                local isDiesel = consumer.fillTypeIndex == FillType.DIESEL

                if isDiesel or isWashing then
                    local consumerUsage = consumer.usagePerMS * dt

                    if not isDiesel then
                        consumerUsage = consumerUsage * pumpWorkMultiplier

                        if consumer.washMultiplierIncrease > 0 and consumer.fillLevel > 0 then
                            washMultiplier = washMultiplier + consumer.washMultiplierIncrease
                        end
                    end

                    self:addFillUnitFillLevel(farmId, consumer.fillUnitIndex, -consumerUsage, consumer.fillTypeIndex, ToolType.UNDEFINED, nil)
                end
            end

            return washMultiplier * pumpWorkMultiplier
        end
    end

    return washMultiplier
end

function PressureWasherVehicle:setPressureWasherActive(isActive)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec.isActive ~= isActive then
        local samples = spec.samples
        local turnOnAnimation = spec.turnOnAnimation

        spec.isActive = isActive

        if isActive then
            spec.startTime = g_currentMission.time + spec.startDuration
            spec.delayedStartTime = spec.startTime + spec.delayedStartDuration

            if self.isClient then
                ObjectChangeUtil.setObjectChanges(spec.objectChanges, true, nil, nil, true)

                g_animationManager:startAnimations(spec.animationNodes)
                g_effectManager:startEffects(spec.effects)

                if samples ~= nil then
                    g_soundManager:stopSample(samples.start)
                    g_soundManager:stopSamples(samples.work)
                    g_soundManager:stopSample(samples.stop)
                    g_soundManager:playSample(samples.start)

                    for i = 1, #samples.work do
                        g_soundManager:playSample(samples.work[i], 0, samples.start)
                    end
                end

                if turnOnAnimation ~= nil and self.playAnimation ~= nil then
                    self:playAnimation(turnOnAnimation.name, turnOnAnimation.turnOnSpeedScale, self:getAnimationTime(turnOnAnimation.name), true)
                end
            end
        else
            spec.isWashing = false

            spec.pump.flowRate = 0
            spec.pump.pressure = 0

            if self.isClient then
                ObjectChangeUtil.setObjectChanges(spec.objectChanges, false, nil, nil, true)

                g_animationManager:stopAnimations(spec.animationNodes)
                g_effectManager:stopEffects(spec.effects)

                if samples ~= nil then
                    g_soundManager:stopSample(samples.start)
                    g_soundManager:stopSamples(samples.work)
                    g_soundManager:stopSample(samples.stop)
                    g_soundManager:playSample(samples.stop)
                end

                if spec.turnOnAnimation ~= nil and self.playAnimation ~= nil then
                    self:playAnimation(turnOnAnimation.name, turnOnAnimation.turnOffSpeedScale, self:getAnimationTime(turnOnAnimation.name), true)
                end
            end
        end
    end

    self:raiseActive()
end

function PressureWasherVehicle:setPressureWasherIsWashing(isWashing)
    self[PressureWasherVehicle.SPEC_NAME].isWashing = isWashing == true
end

function PressureWasherVehicle:getCanPressureWasherRun(checkAllRequired)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec ~= nil and spec.numberOfConsumer > 0 then
        if checkAllRequired then
            for _, consumer in ipairs(spec.consumers) do
                if consumer.isRequired and consumer.fillLevel <= 0 then
                    return false, consumer.emptyWarning
                end
            end
        else
            local consumer = spec.consumersByFillTypeIndex[FillType.DIESEL]

            if consumer ~= nil and consumer.fillLevel <= 0 then
                return false, consumer.emptyWarning
            end
        end

        return spec.startTime < g_currentMission.time, ""
    end

    return true
end

function PressureWasherVehicle:getPressureWasherActive()
    return self[PressureWasherVehicle.SPEC_NAME].isActive
end

function PressureWasherVehicle:getFormattedPressureWasherOperatingTime()
    local minutes = self.operatingTime / (1000 * 60)
    local hours = math.floor(minutes / 60)

    minutes = math.floor((minutes - hours * 60) / 6)
    local minutesString = string.format("%02d", minutes * 10)

    return tonumber(hours .. "." .. minutesString)
end

function PressureWasherVehicle:getPressureWasherEngineLoad()
    return self[PressureWasherVehicle.SPEC_NAME].engineLoad or 0
end

function PressureWasherVehicle:getPressureWasherWorkMultiplier()
    return self[PressureWasherVehicle.SPEC_NAME].pump.workMultiplier or 1
end

function PressureWasherVehicle:loadDashboardGroupFromXML(superFunc, xmlFile, key, group)
    if not superFunc(self, xmlFile, key, group) then
        return false
    end

    group.isPressureWasherRunning = xmlFile:getValue(key .. "#isPressureWasherRunning")
    group.isPressureWasherStarting = xmlFile:getValue(key .. "#isPressureWasherStarting")
    group.isPressureWasherDelayedStart = xmlFile:getValue(key .. "#isPressureWasherDelayedStart")

    return true
end

function PressureWasherVehicle:getIsDashboardGroupActive(superFunc, group)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec ~= nil then
        if group.isPressureWasherRunning and group.isPressureWasherStarting then
            if not spec.isActive then
                return false
            end
        end

        if group.isPressureWasherStarting and not group.isPressureWasherRunning then
            if not spec.isActive or spec.startTime < g_currentMission.time then
                return false
            end
        end

        if group.isPressureWasherRunning and not group.isPressureWasherStarting then
            if not spec.isActive or spec.startTime > g_currentMission.time then
                return false
            end
        end

        if group.isPressureWasherDelayedStart then
            if not spec.isActive or spec.delayedStartTime < g_currentMission.time then
                return false
            end
        end
    end

    return superFunc(self, group)
end

function PressureWasherVehicle:getIsOperating(superFunc)
    return superFunc(self) or self:getPressureWasherActive()
end

function PressureWasherVehicle:showInfo(superFunc, box)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec ~= nil then
        for _, consumer in ipairs (spec.consumers) do
            if consumer.isRequired then
                local fillUnit = self:getFillUnitByIndex(consumer.fillUnitIndex)

                if fillUnit ~= nil and fillUnit.fillLevel <= 0 then
                    box:addLine(consumer.title, string.format("0 %s", fillUnit.unitText or g_i18n:getVolumeUnit()), true)
                end
            end
        end
    end

    superFunc(self, box)
end

function PressureWasherVehicle:updateDebugValues(values)
    local spec = self[PressureWasherVehicle.SPEC_NAME]

    if spec ~= nil and spec.handTool ~= nil then
        local handTool = spec.handTool

        if handTool.getHighPressureWasher ~= nil and handTool:getIsActiveForInput() then
            local lanceSpec = handTool.spec_highPressureWasherLance
            local x, y, z = g_localPlayer:getPosition()

            local vehicle = handTool:getHighPressureWasher()
            local formatedOperatingTime = string.format("%.2f (%d seconds)", vehicle:getFormattedPressureWasherOperatingTime(), (vehicle.operatingTime or 0) / 1000)

            local washMultiplier = lanceSpec.washMultiplier or 0
            local baseWashMultiplier = handTool:getBaseWashMultiplier() or 3
            local pumpWorkMultiplier = vehicle:getPressureWasherWorkMultiplier() or 0

            local debugValues = {
                {name = "Pressure Washer Name", value = vehicle:getName()},
                {name = "Pump Operating Time", value = formatedOperatingTime},
                {name = "Wash Multiplier", value = baseWashMultiplier},
                {name = "Extra Wash Multiplier", value = washMultiplier - baseWashMultiplier},
                {name = "Total Wash Multiplier", value = washMultiplier},
                {name = "Wash Distance", value = math.max(lanceSpec.washDistance * pumpWorkMultiplier, 5)}
            }

            local tethered = handTool.spec_tethered

            if tethered ~= nil then
                local wx, wy, wz = getWorldTranslation(tethered.attachedHolder.holderNode)

                table.insert(debugValues, {name = "Hose Length", value = tethered.maximumRange})
                table.insert(debugValues, {name = "Hose Length Used", value = MathUtil.round(MathUtil.vector3Length(x - wx, y - wy, z - wz))})
            end

            if lanceSpec.targetedVehicleId ~= nil then
                local targetedVehicle = NetworkUtil.getObject(lanceSpec.targetedVehicleId)
                local wx, wy, wz = getWorldTranslation(targetedVehicle.components[1].node)

                table.insert(debugValues, {name = "Vehicle Name", value = targetedVehicle:getName()})
                table.insert(debugValues, {name = "Distance To Vehicle", value = MathUtil.round(MathUtil.vector3Length(x - wx, y - wy, z - wz))})

                if washableSpec ~= nil and Washable.updateDebugValues ~= nil then
                    Washable.updateDebugValues(targetedVehicle, debugValues)
                end
            end

            DebugUtil.renderTable(0.81, 0.65, getCorrectTextSize(0.015), debugValues, 0)
        end
    end
end

g_soundManager:registerModifierType("PRESSURE_WASHER_ENGINE_LOAD", PressureWasherVehicle.getPressureWasherEngineLoad)