--[[
Copyright (C) GtX (Andy), 2020

Author: GtX | Andy
Date: 05.05.2020
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Mod idea / Concept:
Johnny Vee and GtX

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

MultiplayerVehicleKeys = {}

MultiplayerVehicleKeys.MOD_NAME = g_currentModName
MultiplayerVehicleKeys.SPEC_NAME = string.format("%s.multiplayerVehicleKeys", g_currentModName)
MultiplayerVehicleKeys.SPEC_TABLE_NAME = string.format("spec_%s", MultiplayerVehicleKeys.SPEC_NAME)

local playerUniqueUserId = getUniqueUserId()

function MultiplayerVehicleKeys.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Enterable, specializations) and
        SpecializationUtil.hasSpecialization(Motorized, specializations) and
        not SpecializationUtil.hasSpecialization(Locomotive, specializations)
end

function MultiplayerVehicleKeys.initSpecialization()
    local schemaSavegame = Vehicle.xmlSchemaSavegame
    local baseKey = string.format("vehicles.vehicle(?).%s", MultiplayerVehicleKeys.SPEC_NAME)

    schemaSavegame:register(XMLValueType.STRING, baseKey .. "#uniqueUserId", "Unique user ID")
    schemaSavegame:register(XMLValueType.STRING, baseKey .. "#lastNickname", "Last nickname the user had set", "")
end

function MultiplayerVehicleKeys.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanMotorRun", MultiplayerVehicleKeys.getCanMotorRun)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getMotorNotAllowedWarning", MultiplayerVehicleKeys.getMotorNotAllowedWarning)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanStartAIVehicle", MultiplayerVehicleKeys.getCanStartAIVehicle)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getCanStopAIVehicle", MultiplayerVehicleKeys.getCanStopAIVehicle)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "getIsTabbable", MultiplayerVehicleKeys.getIsTabbable)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "showInfo", MultiplayerVehicleKeys.showInfo)
end

function MultiplayerVehicleKeys.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad",                 MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onLoadFinished",         MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onPreDelete",            MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onReadStream",           MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onWriteStream",          MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick",           MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onLeaveVehicle",         MultiplayerVehicleKeys)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", MultiplayerVehicleKeys)
end

function MultiplayerVehicleKeys.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "addKeyOwner", MultiplayerVehicleKeys.addKeyOwner)
    SpecializationUtil.registerFunction(vehicleType, "removeKeyOwner", MultiplayerVehicleKeys.removeKeyOwner)
    SpecializationUtil.registerFunction(vehicleType, "setKeyOwner", MultiplayerVehicleKeys.setKeyOwner)
    SpecializationUtil.registerFunction(vehicleType, "getHasKeyOwner", MultiplayerVehicleKeys.getHasKeyOwner)
    SpecializationUtil.registerFunction(vehicleType, "getIsKeyOwner", MultiplayerVehicleKeys.getIsKeyOwner)
    SpecializationUtil.registerFunction(vehicleType, "getIsVehicleKeyAvailable", MultiplayerVehicleKeys.getIsVehicleKeyAvailable)
end

function MultiplayerVehicleKeys:onLoad(savegame)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]
    local customEnvironment = self.customEnvironment

    spec.keyOwnerUniqueUserId = ""
    spec.keyOwnerLastNickname = ""
    spec.isKeyOwner = false

    spec.holdingKeysInfo = g_i18n:getText("info_holdingKeys", customEnvironment)
    spec.holdingKeysTitle = g_i18n:getText("info_holdingKeysTitle", customEnvironment)
    spec.keysReturnedInfo = g_i18n:getText("info_keysReturned", customEnvironment)

    spec.actionRemoveKey = g_i18n:getText("action_removeVehicleKey", customEnvironment)
    spec.actionReturnKey = g_i18n:getText("action_returnVehicleKey", customEnvironment)

    spec.invalidUserWarning = string.format("%s\n%s", g_i18n:getText("warning_youDontHaveAccessToThis"), g_i18n:getText("info_keyOwner", customEnvironment))
    spec.holdingKeysWarning = string.format("%s\n%s: ", g_i18n:getText("warning_actionNotAllowedNow"), spec.holdingKeysTitle) .. "%{numKeys}d/%{keyLimit}d"

    self.spec_enterableMultiplayerKeys = spec
end

function MultiplayerVehicleKeys:onLoadFinished(savegame)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if savegame ~= nil then
        local savegameKey = savegame.key .. "." .. MultiplayerVehicleKeys.SPEC_NAME
        local uniqueUserId = savegame.xmlFile:getValue(savegameKey .. "#uniqueUserId")

        if uniqueUserId ~= nil then
            local lastNickname = savegame.xmlFile:getValue(savegameKey .. "#lastNickname", "")

            if playerUniqueUserId == uniqueUserId then
                self:setKeyOwner(uniqueUserId, g_currentMission.playerNickname or lastNickname, false)
            else
                self:setKeyOwner(uniqueUserId, lastNickname, false)
            end
        end
    end
end

function MultiplayerVehicleKeys:onPreDelete()
    self:removeKeyOwner(false, true)

    if g_multiplayerVehicleKeys ~= nil then
        g_multiplayerVehicleKeys:removeKey(self)
    end
end

function MultiplayerVehicleKeys:onReadStream(streamId, connection)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if streamReadBool(streamId) then
        local uniqueUserId = streamReadString(streamId)
        local lastNickname = streamReadString(streamId)

        self:setKeyOwner(uniqueUserId, lastNickname, false)
    end
end

function MultiplayerVehicleKeys:onWriteStream(streamId, connection)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if streamWriteBool(streamId, spec.keyOwnerUniqueUserId ~= "") then
        streamWriteString(streamId, spec.keyOwnerUniqueUserId)
        streamWriteString(streamId, spec.keyOwnerLastNickname or "Unknown Player")
    end
end

function MultiplayerVehicleKeys:onUpdateTick(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    if self.isClient and isActiveForInputIgnoreSelection then
        local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]
        local actionEvent = spec.actionEvents[InputAction.VEHICLE_KEY_INTERACT]

        if actionEvent ~= nil then
            if self:getIsMotorStarted() then
                g_inputBinding:setActionEventTextPriority(actionEvent.actionEventId, GS_PRIO_LOW)
            else
                g_inputBinding:setActionEventTextPriority(actionEvent.actionEventId, GS_PRIO_VERY_HIGH)
            end

            g_inputBinding:setActionEventText(actionEvent.actionEventId, spec.keyOwnerUniqueUserId == "" and spec.actionRemoveKey or spec.actionReturnKey)
        end
    end
end

function MultiplayerVehicleKeys:onLeaveVehicle()
    local blinkingWarning = g_currentMission.hud.blinkingWarning

    if blinkingWarning ~= nil and blinkingWarning.duration > 0 then
        blinkingWarning.duration = -1
    end
end

function MultiplayerVehicleKeys:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if isActiveForInputIgnoreSelection then
        self:clearActionEventsTable(spec.actionEvents)

        if (spec.keyOwnerUniqueUserId == "" or spec.keyOwnerUniqueUserId == playerUniqueUserId) and self:getIsActiveForInput(true, true) then
            local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.VEHICLE_KEY_INTERACT, self, MultiplayerVehicleKeys.actionEventVehicleKeyInteract, false, true, false, true, nil)

            g_inputBinding:setActionEventText(actionEventId, spec.keyOwnerUniqueUserId == "" and spec.actionRemoveKey or spec.actionReturnKey)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_LOW)
        end
    end
end

function MultiplayerVehicleKeys:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if spec ~= nil and spec.keyOwnerUniqueUserId ~= "" then
        xmlFile:setValue(key .. "#uniqueUserId", spec.keyOwnerUniqueUserId)
        xmlFile:setValue(key .. "#lastNickname", spec.keyOwnerLastNickname or "Unknown Player")
    end
end

function MultiplayerVehicleKeys:addKeyOwner(userId, showNotification, noEventSend)
    local user = g_currentMission.userManager:getUserByUserId(userId)

    if user ~= nil then
        MultiplayerVehicleKeysSetOwnerEvent.sendEvent(self, userId, showNotification, noEventSend)

        self:setKeyOwner(user:getUniqueUserId(), user:getNickname(), showNotification)
    end
end

function MultiplayerVehicleKeys:removeKeyOwner(showNotification, noEventSend)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    MultiplayerVehicleKeysSetOwnerEvent.sendEvent(self, nil, showNotification, noEventSend)

    if spec.keyOwnerUniqueUserId == playerUniqueUserId then
        if g_multiplayerVehicleKeys ~= nil then
            g_multiplayerVehicleKeys:removeKey(self)
        end

        if showNotification then
            g_currentMission.hud:addSideNotification(FSBaseMission.INGAME_NOTIFICATION_INFO, string.namedFormat(spec.keysReturnedInfo, "vehicleName", tostring(self:getFullName())), 4000)
        end
    end

    spec.keyOwnerUniqueUserId = ""
    spec.keyOwnerLastNickname = ""
    spec.isKeyOwner = false

    if self.isClient and self:getIsEntered() then
        self:requestActionEventUpdate()
    end
end

function MultiplayerVehicleKeys:setKeyOwner(uniqueUserId, nickname, showNotification)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    spec.keyOwnerUniqueUserId = uniqueUserId or ""
    spec.keyOwnerLastNickname = nickname or "Unknown Player"

    spec.isKeyOwner = uniqueUserId == playerUniqueUserId

    if spec.isKeyOwner then
        if g_multiplayerVehicleKeys ~= nil then
            g_multiplayerVehicleKeys:addKey(self)
        end

        if showNotification then
            g_currentMission.hud:addSideNotification(FSBaseMission.INGAME_NOTIFICATION_OK, string.namedFormat(spec.holdingKeysInfo, "vehicleName", tostring(self:getFullName())), 4000)
        end
    end
end

function MultiplayerVehicleKeys:actionEventVehicleKeyInteract(actionName, inputValue, callbackState, isAnalog)
    if self:getIsEntered() and g_multiplayerVehicleKeys ~= nil then
        local currentVehicleKeys = g_multiplayerVehicleKeys:getCurrentKeys()

        if currentVehicleKeys ~= nil then
            if currentVehicleKeys[self] then
                self:removeKeyOwner(true)
            else
                local numKeys = table.size(currentVehicleKeys)
                local keyLimit = g_multiplayerVehicleKeys:getKeyLimit()

                if numKeys < keyLimit then
                    self:addKeyOwner(g_currentMission.playerUserId, true)
                else
                    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]
                    local warning = string.namedFormat(spec.holdingKeysWarning, "numKeys", numKeys, "keyLimit", keyLimit)

                    g_currentMission:showBlinkingWarning(warning, 4000)
                end
            end
        end
    end
end

function MultiplayerVehicleKeys:getHasKeyOwner()
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    return spec ~= nil and spec.keyOwnerUniqueUserId ~= ""
end

function MultiplayerVehicleKeys:getIsKeyOwner(uniqueUserId)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    return spec ~= nil and spec.keyOwnerUniqueUserId == (uniqueUserId or playerUniqueUserId)
end

function MultiplayerVehicleKeys:getIsVehicleKeyAvailable()
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if spec == nil or spec.keyOwnerUniqueUserId == "" or spec.keyOwnerUniqueUserId == playerUniqueUserId then
        return true
    end

    if self:getIsAIActive() then
        return true
    end

    local user = g_currentMission.userManager:getUserByUserId(self.spec_enterable.controllerUserId)

    if user ~= nil then
        if spec.keyOwnerUniqueUserId == user:getUniqueUserId() then
            return true
        end

        if user:getIsMasterUser() and (g_multiplayerVehicleKeys == nil or g_multiplayerVehicleKeys:getMasterKeyEnabled()) then
            return g_currentMission.userManager:getUserByUniqueId(spec.keyOwnerUniqueUserId) == nil
        end
    end

    return false
end

function MultiplayerVehicleKeys:showInfo(superFunc, box)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if spec ~= nil and spec.keyOwnerUniqueUserId ~= "" and spec.keyOwnerLastNickname ~= "" then
        box:addLine(spec.holdingKeysTitle, spec.keyOwnerLastNickname or "Unknown Player")
    end

    superFunc(self, box)
end

function MultiplayerVehicleKeys:getCanMotorRun(superFunc, ...)
    return self:getIsVehicleKeyAvailable() and superFunc(self, ...)
end

function MultiplayerVehicleKeys:getMotorNotAllowedWarning(superFunc)
    if not self:getIsVehicleKeyAvailable() then
        local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

        return string.namedFormat(spec.invalidUserWarning, "playerName", spec.keyOwnerLastNickname or "Unknown Player")
    end

    return superFunc(self)
end

function MultiplayerVehicleKeys:getCanStartAIVehicle(superFunc, ...)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if spec ~= nil and spec.keyOwnerUniqueUserId ~= "" and spec.keyOwnerUniqueUserId ~= playerUniqueUserId then
        return false
    end

    return superFunc(self, ...)
end

function MultiplayerVehicleKeys:getCanStopAIVehicle(superFunc, ...)
    local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

    if spec ~= nil and spec.keyOwnerUniqueUserId ~= "" and spec.keyOwnerUniqueUserId ~= playerUniqueUserId then
        -- Allow an admin or host to disable the AI in-case of emergency but not start them
        if not (self.isServer or g_currentMission.isMasterUser) then
            return false
        end
    end

    return superFunc(self, ...)
end

function MultiplayerVehicleKeys:getIsTabbable(superFunc)
    if superFunc(self) then
        if g_dedicatedServer == nil then
            local spec = self[MultiplayerVehicleKeys.SPEC_TABLE_NAME]

            if spec ~= nil and spec.keyOwnerUniqueUserId ~= "" and not spec.savingStats then
                return spec.keyOwnerUniqueUserId == playerUniqueUserId -- Key owner is set, return whether the person trying to enter is the key holder
            end
        end

        return true -- No key owner, dedicated server is saving stats or spec is missing for some reason
    end

    return false -- The vehicle is set as non tabbable or another mod has changed the state such as a mod that can park vehicles (Circa FS15)
end
