KommunalMission = {
	REWARD_PER_HA = 37500,	  -- soldi per ettaro
	NAME = "kommunalMission"
}

local KommunalMission_mt = Class(KommunalMission, AbstractMission)
InitObjectClass(KommunalMission, "KommunalMission")

function KommunalMission.new(isServer, isClient, customMt)
	local area_tot = 500 + math.random()*200
	local area_truncated = tonumber(string.format("%.0f", area_tot))
	local description = string.format(g_i18n:getText("fieldJob_desc_kommunal"), area_truncated)
    local title = g_i18n:getText("fieldJob_jobType_kommunal")
	
	local self = AbstractMission.new(isServer, isClient, title, description, customMt or KommunalMission_mt)

	self.area_trimmed = 0
    self.npc_idx = g_npcManager:getRandomIndex()
	self.area_tot = area_tot		-- area totale da tagliare (in metri)

    self.field = {
        fieldId = 0,
        getName = function() return "" end,
        getAreaHa = function () return 1 end
    }

    return self
end

function KommunalMission:init()
    return KommunalMission:superClass().init(self)
end

function KommunalMission:saveToXMLFile(xmlFile, key)
	KommunalMission:superClass().saveToXMLFile(self, xmlFile, key)

	local kommunalKey = string.format("%s.info", key)
	
    xmlFile:setValue(kommunalKey .. "#totalArea", self.area_tot)
    xmlFile:setValue(kommunalKey .. "#areaTrimmed", self.area_trimmed)
    xmlFile:setValue(kommunalKey .. "#npcIndex", self.npc_idx)
end


---Load a mission from the savegame. Subclass to add custom properties.
function KommunalMission:loadFromXMLFile(xmlFile, key)
	if not KommunalMission:superClass().loadFromXMLFile(self, xmlFile, key) then
		return false end
	
	local kommunalKey = string.format("%s.info", key)
    
    self.area_tot = xmlFile:getValue(kommunalKey .. "#totalArea")
    self.area_trimmed = xmlFile:getValue(kommunalKey .. "#areaTrimmed")
    self.npc_idx = xmlFile:getValue(kommunalKey .. "#npcIndex")
	return true
end



---Write the mission to a stream
function KommunalMission:writeStream(streamId, connection)
    KommunalMission:superClass().writeStream(self, streamId, connection)

    streamWriteFloat32(streamId, self.area_tot)
    streamWriteUInt8(streamId, self.npc_idx)
end


---Read the mission from a stream
function KommunalMission:readStream(streamId, connection)
    KommunalMission:superClass().readStream(self, streamId, connection)

    self.area_tot = streamReadFloat32(streamId)
    self.npc_idx = streamReadUInt8(streamId)
end

---Write updates of a running mission to a stream
function KommunalMission:writeUpdateStream(streamId, connection)
    KommunalMission:superClass().writeUpdateStream(self, streamId, connection)

    if streamWriteBool(streamId, self.area_trimmed > 0) then
        streamWriteFloat32(streamId, self.area_trimmed)
    end
end

---Read updates of a running mission from stream
function KommunalMission:readUpdateStream(streamId, connection)
    KommunalMission:superClass().readUpdateStream(self, streamId, connection)

    if streamReadBool(streamId) then
        self.area_trimmed = streamReadFloat32(streamId) or 0
    end
end


function KommunalMission:getReward()
    -- 1 ettaro = 100 m * 100 m = 10^4 m^2
    return KommunalMission.REWARD_PER_HA * self.area_tot / 10000;
end

-- Get the NPC giving this mission
function KommunalMission:getNPC()
    return g_npcManager:getNPCByIndex(self.npc_idx)
end

function KommunalMission:getDetails()
    -- body
    local list = KommunalMission:superClass().getDetails(self)
    table.insert(list, {
        title = string.format("%s %s", g_i18n:getText("contract_total"),
                g_i18n:getText("ui_fieldArea")),
        value = g_i18n:formatNumber(self.area_tot) .. " m2"
    })
    if self.status == MissionStatus.RUNNING then
        table.insert(list, {
            title = g_i18n:getText("contract_kommunal_trimmed"),
            value = g_i18n:formatNumber(self.area_trimmed) .. " m2"
            }
        )
        table.insert(list, {
            title = g_i18n:getText("contract_kommunal_remaining"),
            value = g_i18n:formatNumber(self.area_tot-self.area_trimmed) .. " m2"
            }
        )
    end
    return list
end


function KommunalMission:getLocation()
    return g_i18n:getText("fieldJob_location_kommunal")
end

function KommunalMission:getVehicleSize()
    if self.v_size ~= nil then
        return self.v_size
    end

    local size_idx = (self.area_tot - 500) / 200
    local size = "medium"
    if size_idx < 0.33 then
        size = "small"
    elseif size_idx < 0.66 then
        size = "medium"
    else
        size = "large"
    end
    self.v_size = size
    return size
end

function KommunalMission:getExtraProgressText()
    return string.format(g_i18n:getText("fieldJob_extra_progress_text_kommunal"), 
         g_i18n:formatNumber(self.area_trimmed), 
         g_i18n:formatNumber(self.area_tot - self.area_trimmed))
end

---Get the completion percentage. When 1.0, mission succeeds and finis  hes.
function KommunalMission:getCompletion()
	return math.min(1, self.area_trimmed/self.area_tot)
end

function KommunalMission:getMissionTypeName()
    return KommunalMission.NAME
end

--- TODO: controllare che il terreno non sia già dell'utente
function KommunalMission:getIsWorkAllowed(farmId, x, z, workAreaType, vehicle)
    return true
end

function KommunalMission.tryGenerateMission()
    if KommunalMission.canRun() then
		local mission = KommunalMission.new(true, g_client ~= nil)
		if mission:init() then
			mission:setDefaultEndDate()
			return mission
		else
			mission:delete()
		end
	end

	return nil
end

function KommunalMission:registerTrimmedArea(area)
	self.area_trimmed = self.area_trimmed + area
end

function KommunalMission.canRun()
	local data = g_missionManager:getMissionTypeDataByName(KommunalMission.NAME)

    if data.numInstances >= data.maxNumInstances then
        return false
    end

    return true
end


function PrerequisitesCheck()
	local bc_ModName = "FS25_BetterContracts"
	local bc_Ver = "1.1.1.5"
	if g_modIsLoaded[bc_ModName] then
		if g_modManager:getModByName(bc_ModName).version < bc_Ver then      
			Logging.error("FS25_KommunalServices is incompatible with BetterContracts versions below %s! The mod will shut down.", bc_Ver)
			InfoDialog.show(string.format(g_i18n:getText("better_contracts_conflict"), bc_Ver))
			return false
		end
	end 
	local cp_modName = "FS25_Contracts_Plus"
	if g_modIsLoaded[cp_modName] then
		Logging.error("FS25_KommunalServices is incompatible with Contracts plus! The mod will shut down.")
		InfoDialog.show(g_i18n:getText("contracts_plus_conflict"))
		return false
	end
	return true
end

MissionManager.loadMapData = 
    Utils.prependedFunction(
        MissionManager.loadMapData,
        function(xmlFile)
            if (PrerequisitesCheck() == false) then
                return end
            if g_missionManager:getMissionTypeDataByName(KommunalMission.NAME) == nil then
                g_missionManager:registerMissionType(KommunalMission, KommunalMission.NAME, 30) -- ultimo arg è defaultMaxNumInstances
            end

            MissionManager.xmlSchemaSavegame:register(XMLValueType.STRING, "missions.kommunalMission(?)#uniqueId", "The unique Id of the mission")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.INT, "missions.kommunalMission(?)#farmId", "Farm ID")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.INT, "missions.kommunalMission(?)#activeId", "Active ID")
            
            MissionManager.xmlSchemaSavegame:register(XMLValueType.FLOAT, "missions.kommunalMission(?).info#reward", "Reward")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.FLOAT, "missions.kommunalMission(?).info#reimbursement", "Reimbursement")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.FLOAT, "missions.kommunalMission(?).info#completion", "Completion")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.FLOAT, "missions.kommunalMission(?).info#stealingCost", "Stealing cost")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.FLOAT, "missions.kommunalMission(?).info#totalArea", "Total area")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.FLOAT, "missions.kommunalMission(?).info#areaTrimmed", "Total area already trimmed (in squared meters)")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.INT, "missions.kommunalMission(?).info#npcIndex", "Index ofthe mission's npc")

            MissionManager.xmlSchemaSavegame:register(XMLValueType.BOOL, "missions.kommunalMission(?).vehicles#spawned", "Whether the necessary vehicles have been spawned")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.INT, "missions.kommunalMission(?).vehicles#group", "The group of the vehicle")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.STRING, "missions.kommunalMission(?).vehicles.vehicle(?)#uniqueId", "The unique ID of the vehicle")

            MissionManager.xmlSchemaSavegame:register(XMLValueType.INT, "missions.kommunalMission(?).endDate#endDay", "The day in which the contract ends")
            MissionManager.xmlSchemaSavegame:register(XMLValueType.INT, "missions.kommunalMission(?).endDate#endDayTime", "The day in which the contract ends")
        end
)
